// Background service worker for Divinge extension
// Handles cookie access and monitoring

const POE_DOMAIN = 'pathofexile.com';

// Listen for messages from popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'getCookies') {
    getCookies().then(sendResponse);
    return true; // Keep channel open for async response
  }
});

// Get PoE cookies - try multiple URL patterns
async function getCookies() {
  try {
    // Try different URL patterns as cookies might be set on different subdomains
    const urls = [
      'https://www.pathofexile.com',
      'https://pathofexile.com',
    ];

    let poesessid = null;
    let cfClearance = null;

    for (const url of urls) {
      if (!poesessid) {
        const cookie = await chrome.cookies.get({ url, name: 'POESESSID' });
        if (cookie?.value) poesessid = cookie.value;
      }
      if (!cfClearance) {
        const cookie = await chrome.cookies.get({ url, name: 'cf_clearance' });
        if (cookie?.value) cfClearance = cookie.value;
      }
    }

    // Also try getting all cookies for the domain to debug
    const allCookies = await chrome.cookies.getAll({ domain: 'pathofexile.com' });
    console.log('All PoE cookies:', allCookies.map(c => ({ name: c.name, domain: c.domain })));

    return { poesessid, cfClearance };
  } catch (err) {
    console.error('Failed to get cookies:', err);
    return { poesessid: null, cfClearance: null };
  }
}

// Optional: Monitor cookie changes and notify user
chrome.cookies.onChanged.addListener((changeInfo) => {
  const { cookie, removed } = changeInfo;

  // Only care about pathofexile.com cookies
  if (!cookie.domain.includes(POE_DOMAIN)) return;

  // Only care about our specific cookies
  if (cookie.name !== 'POESESSID' && cookie.name !== 'cf_clearance') return;

  if (removed) {
    console.log(`PoE cookie removed: ${cookie.name}`);
    // Could show a notification here
  } else {
    console.log(`PoE cookie updated: ${cookie.name}`);
    // Could auto-sync here if enabled
  }
});

// Set up alarm for periodic cookie check (optional feature for later)
// chrome.alarms.create('checkCookies', { periodInMinutes: 30 });
// chrome.alarms.onAlarm.addListener((alarm) => {
//   if (alarm.name === 'checkCookies') {
//     // Check and possibly auto-sync
//   }
// });
